<?php if (!defined('PmWiki')) exit();
/*
-----------------------------------------------------------------------------------
      * Cookbook Function for counting string occurrences *
                     Written by (c) Holger Kremb 2026
        Tested with PHP 8.5.2 - last tested on PmWiki version: 2.5.5

  This text is written for PmWiki; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3 of the License, or (at your option) any later
  version. See pmwiki.php for full details and lack of warranty.
-----------------------------------------------------------------------------------
  Usage:
    (:countoccurrences needle=test:)

  Options:
    ignorecase=1     Case-insensitive matching
    wholeword=1      Match whole words only
    start= / end=    Limit counting to marker range
    id=              Store result in a PageVar
    get=             Read stored PageVar without recount

  Marker example:
    (:countoccurrences needle=test start="[[#Pm_A]]" end="[[#Pm_B]]":)

  Cached value:
    (:countoccurrences needle=test id=c01:)
    (:countoccurrences get=c01:)

  Conditional usage:
    (:if expr ( (:countoccurrences get=c01:) ) > 10 :)
      Too many hits.
    (:ifend:)

  Alias (compatibility):
    (:pmcountstr ...:)

  Note:
    "Pm_" must precede the actual marker name to prevent marker duplication
    when anchors are rendered by PmWiki.
-----------------------------------------------------------------------------------
  Cookbook file : countoccurrences.php
  Markup        : (:countoccurrences:)  (alias: :pmcountstr:)
  PHP function  : CountOccurrences()

  Copyright 2026 Holger Kremb
  https://kremb.net
-----------------------------------------------------------------------------------
  Acknowledgement:
    This recipe is based on an idea by "nitram", discussed on the PmWiki
    Cookbook Talk pages for CountBetweenMarkers. The challenge inspired
    the implementation of CountOccurrences.

    Challenges are there to be taken on.
-----------------------------------------------------------------------------------
*/

$RecipeInfo['CountOccurrences']['Version'] = '20260227';
$FmtPV['$CountOccurrences'] = "'CO-{$RecipeInfo['CountOccurrences']['Version']}'";

/* request-local cache for "id"/"get" use */
$GLOBALS['CountOccurrencesCache'] = $GLOBALS['CountOccurrencesCache'] ?? array();

function CountOccurrences($m) {
  global $pagename, $FmtPV;

  $opt = ParseArgs($m[1]);

  /* sanitize id/get keys */
  $id  = isset($opt['id'])  ? preg_replace('/[^A-Za-z0-9_]/', '', (string)$opt['id'])  : '';
  $get = isset($opt['get']) ? preg_replace('/[^A-Za-z0-9_]/', '', (string)$opt['get']) : '';

  /* getter mode: (:countoccurrences get=c01:) or (:countoccurrences id=c01 get=1:) */
  if ($get !== '' || (!empty($opt['get']) && $id !== '')) {
    $key = ($get !== '') ? $get : $id;
    $val = 0;
    if (isset($GLOBALS['CountOccurrencesCache'][$key])) {
      $val = (int)$GLOBALS['CountOccurrencesCache'][$key];
    } elseif (isset($FmtPV['$'.$key])) {
      /* fallback if already set somewhere */
      $raw = (string)$FmtPV['$'.$key];
      $raw = trim($raw, " \t\n\r\0\x0B'\"");
      if ($raw !== '' && preg_match('/^-?\d+$/', $raw)) $val = (int)$raw;
    }
    return (string)$val;
  }

  $page = (!empty($opt['page'])) ? MakePageName($pagename, (string)$opt['page']) : $pagename;

  $needle = isset($opt['needle']) ? (string)$opt['needle'] : '';
  if ($needle === '') {
    if ($id !== '') {
      $GLOBALS['CountOccurrencesCache'][$id] = 0;
      $FmtPV['$'.$id] = "'0'";
    }
    return '0';
  }

  $ignorecase = !empty($opt['ignorecase']);
  $wholeword  = !empty($opt['wholeword']);

  $start = isset($opt['start']) ? (string)$opt['start'] : '';
  $end   = isset($opt['end'])   ? (string)$opt['end']   : '';

  if ($start !== '' && preg_match('/^\s*\[\[\s*#Pm_([^\]\s]+)\s*\]\]\s*$/', $start, $mm)) $start = '#'.$mm[1];
  if ($end   !== '' && preg_match('/^\s*\[\[\s*#Pm_([^\]\s]+)\s*\]\]\s*$/', $end,   $mm)) $end   = '#'.$mm[1];

  $p = ReadPage($page, READPAGE_CURRENT);
  $txt = isset($p['text']) ? (string)$p['text'] : '';

  if ($txt === '') {
    if ($id !== '') {
      $GLOBALS['CountOccurrencesCache'][$id] = 0;
      $FmtPV['$'.$id] = "'0'";
    }
    return '0';
  }

  $lines = preg_split("/\r\n|\n|\r/", $txt);

  if ($start !== '' && $end !== '') {
    $in = false;
    $buf = array();
    foreach ($lines as $ln) {
      if (!$in) {
        if (strpos($ln, $start) !== false) $in = true;
        continue;
      }
      if (strpos($ln, $end) !== false) break;
      $buf[] = $ln;
    }
    $txt = implode("\n", $buf);
  }

  if ($wholeword) {
    $pat = '/\b' . preg_quote($needle, '/') . '\b/' . ($ignorecase ? 'i' : '');
    $count = preg_match_all($pat, $txt, $dummy);
    if ($count === false) $count = 0;
  } else {
    if ($ignorecase) {
      $count = substr_count(strtolower($txt), strtolower($needle));
    } else {
      $count = substr_count($txt, $needle);
    }
  }

  $count = (int)$count;

  if ($id !== '') {
    $GLOBALS['CountOccurrencesCache'][$id] = $count;
    $FmtPV['$'.$id] = "'".$count."'";
  }

  return (string)$count;
}

# Markup: primary
Markup_e(
  'countoccurrences',
  'directives',
  '/\\(:countoccurrences\\s*(.*?)\\s*:\\)/',
  'CountOccurrences'
);

# Markup: alias
Markup_e(
  'pmcountstr',
  'directives',
  '/\\(:pmcountstr\\s*(.*?)\\s*:\\)/',
  'CountOccurrences'
);